<?php
require_once 'config.php';

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

switch ($method) {
    case 'GET':
        handleGetRequests();
        break;
    case 'POST':
        handlePostRequests();
        break;
    case 'PUT':
        handlePutRequests();
        break;
    default:
        sendResponse(['error' => 'Method not allowed'], 405);
}

function handleGetRequests() {
    global $pdo;
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'messages':
            getMessages();
            break;
        case 'active_chats':
            getActiveChats();
            break;
        case 'chat_history':
            getChatHistory();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

function handlePostRequests() {
    global $input;
    $action = $input['action'] ?? '';
    
    switch ($action) {
        case 'send_message':
            sendMessage();
            break;
        case 'start_chat':
            startChat();
            break;
        case 'close_chat':
            closeChat();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

function handlePutRequests() {
    global $input;
    $action = $input['action'] ?? '';
    
    switch ($action) {
        case 'update_status':
            updateChatStatus();
            break;
        case 'mark_read':
            markMessagesRead();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

// Get messages for a specific chat
function getMessages() {
    global $pdo;
    
    $chatId = $_GET['chat_id'] ?? '';
    $limit = $_GET['limit'] ?? 50;
    $offset = $_GET['offset'] ?? 0;
    
    if (empty($chatId)) {
        sendResponse(['error' => 'Chat ID is required'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT m.*, u.username, u.email
            FROM chat_messages m
            LEFT JOIN users u ON m.user_id = u.id
            WHERE m.chat_id = ?
            ORDER BY m.created_at DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$chatId, $limit, $offset]);
        $messages = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $messages
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch messages'], 500);
    }
}

// Get active chats (for admin)
function getActiveChats() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                cs.id,
                cs.user_id,
                cs.status,
                cs.created_at,
                cs.updated_at,
                u.username,
                u.email,
                COUNT(cm.id) as message_count,
                MAX(cm.created_at) as last_message_at
            FROM chat_sessions cs
            LEFT JOIN users u ON cs.user_id = u.id
            LEFT JOIN chat_messages cm ON cs.id = cm.chat_id
            WHERE cs.status IN ('active', 'pending')
            GROUP BY cs.id
            ORDER BY cs.updated_at DESC
        ");
        $stmt->execute();
        $chats = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $chats
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch active chats'], 500);
    }
}

// Get chat history for a user
function getChatHistory() {
    global $pdo;
    
    $userId = $_GET['user_id'] ?? '';
    
    if (empty($userId)) {
        sendResponse(['error' => 'User ID is required'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                cs.*,
                COUNT(cm.id) as message_count,
                MAX(cm.created_at) as last_message_at
            FROM chat_sessions cs
            LEFT JOIN chat_messages cm ON cs.id = cm.chat_id
            WHERE cs.user_id = ?
            GROUP BY cs.id
            ORDER BY cs.created_at DESC
        ");
        $stmt->execute([$userId]);
        $history = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $history
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch chat history'], 500);
    }
}

// Send a message
function sendMessage() {
    global $pdo, $input;
    
    validateRequired($input, ['chat_id', 'message', 'sender_type']);
    
    $chatId = $input['chat_id'];
    $message = trim($input['message']);
    $senderType = $input['sender_type']; // 'user' or 'admin'
    $userId = $input['user_id'] ?? null;
    $adminId = $input['admin_id'] ?? null;
    
    if (empty($message)) {
        sendResponse(['error' => 'Message cannot be empty'], 400);
    }
    
    try {
        $pdo->beginTransaction();
        
        // Insert message
        $messageId = generateId('msg_');
        $stmt = $pdo->prepare("
            INSERT INTO chat_messages (id, chat_id, user_id, admin_id, message, sender_type, created_at)
            VALUES (?, ?, ?, ?, ?, ?, NOW())
        ");
        $stmt->execute([$messageId, $chatId, $userId, $adminId, $message, $senderType]);
        
        // Update chat session timestamp
        $stmt = $pdo->prepare("UPDATE chat_sessions SET updated_at = NOW() WHERE id = ?");
        $stmt->execute([$chatId]);
        
        $pdo->commit();
        
        sendResponse([
            'success' => true,
            'message_id' => $messageId,
            'message' => 'Message sent successfully'
        ]);
    } catch(PDOException $e) {
        $pdo->rollback();
        sendResponse(['error' => 'Failed to send message'], 500);
    }
}

// Start a new chat session
function startChat() {
    global $pdo, $input;
    
    validateRequired($input, ['user_id', 'subject']);
    
    $userId = $input['user_id'];
    $subject = $input['subject'];
    $priority = $input['priority'] ?? 'normal';
    
    try {
        // Check if user has an active chat
        $stmt = $pdo->prepare("
            SELECT id FROM chat_sessions 
            WHERE user_id = ? AND status IN ('active', 'pending')
        ");
        $stmt->execute([$userId]);
        $existingChat = $stmt->fetch();
        
        if ($existingChat) {
            sendResponse([
                'success' => true,
                'chat_id' => $existingChat['id'],
                'message' => 'Using existing chat session'
            ]);
        }
        
        // Create new chat session
        $chatId = generateId('chat_');
        $stmt = $pdo->prepare("
            INSERT INTO chat_sessions (id, user_id, subject, priority, status, created_at, updated_at)
            VALUES (?, ?, ?, ?, 'pending', NOW(), NOW())
        ");
        $stmt->execute([$chatId, $userId, $subject, $priority]);
        
        sendResponse([
            'success' => true,
            'chat_id' => $chatId,
            'message' => 'Chat session started successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to start chat session'], 500);
    }
}

// Close a chat session
function closeChat() {
    global $pdo, $input;
    
    validateRequired($input, ['chat_id']);
    
    $chatId = $input['chat_id'];
    $closedBy = $input['closed_by'] ?? 'user';
    
    try {
        $stmt = $pdo->prepare("
            UPDATE chat_sessions 
            SET status = 'closed', closed_by = ?, closed_at = NOW(), updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$closedBy, $chatId]);
        
        sendResponse([
            'success' => true,
            'message' => 'Chat session closed successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to close chat session'], 500);
    }
}

// Update chat status
function updateChatStatus() {
    global $pdo, $input;
    
    validateRequired($input, ['chat_id', 'status']);
    
    $chatId = $input['chat_id'];
    $status = $input['status'];
    $adminId = $input['admin_id'] ?? null;
    
    $validStatuses = ['pending', 'active', 'closed'];
    if (!in_array($status, $validStatuses)) {
        sendResponse(['error' => 'Invalid status'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            UPDATE chat_sessions 
            SET status = ?, admin_id = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$status, $adminId, $chatId]);
        
        sendResponse([
            'success' => true,
            'message' => 'Chat status updated successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to update chat status'], 500);
    }
}

// Mark messages as read
function markMessagesRead() {
    global $pdo, $input;
    
    validateRequired($input, ['chat_id']);
    
    $chatId = $input['chat_id'];
    $readBy = $input['read_by'] ?? 'user';
    
    try {
        $stmt = $pdo->prepare("
            UPDATE chat_messages 
            SET read_at = NOW() 
            WHERE chat_id = ? AND read_at IS NULL AND sender_type != ?
        ");
        $stmt->execute([$chatId, $readBy]);
        
        sendResponse([
            'success' => true,
            'message' => 'Messages marked as read'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to mark messages as read'], 500);
    }
}
?>