<?php
require_once 'config.php';

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

switch ($method) {
    case 'GET':
        handleGetRequests();
        break;
    case 'POST':
        handlePostRequests();
        break;
    case 'PUT':
        handlePutRequests();
        break;
    default:
        sendResponse(['error' => 'Method not allowed'], 405);
}

function handleGetRequests() {
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'user_balance':
            getUserBalance();
            break;
        case 'deposit_history':
            getDepositHistory();
            break;
        case 'pending_deposits':
            getPendingDeposits();
            break;
        case 'payment_methods':
            getPaymentMethods();
            break;
        case 'btc_rates':
            getBtcRates();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

function handlePostRequests() {
    global $input;
    $action = $input['action'] ?? '';
    
    switch ($action) {
        case 'create_deposit':
            createDeposit();
            break;
        case 'confirm_deposit':
            confirmDeposit();
            break;
        case 'process_payment':
            processPayment();
            break;
        case 'generate_btc_address':
            generateBtcAddress();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

function handlePutRequests() {
    global $input;
    $action = $input['action'] ?? '';
    
    switch ($action) {
        case 'update_deposit_status':
            updateDepositStatus();
            break;
        case 'update_balance':
            updateUserBalance();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

// Get user balance
function getUserBalance() {
    global $pdo;
    
    $userId = $_GET['user_id'] ?? '';
    
    if (empty($userId)) {
        sendResponse(['error' => 'User ID is required'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                balance,
                pending_balance,
                total_deposited,
                total_withdrawn
            FROM users 
            WHERE id = ?
        ");
        $stmt->execute([$userId]);
        $balance = $stmt->fetch();
        
        if (!$balance) {
            sendResponse(['error' => 'User not found'], 404);
        }
        
        sendResponse([
            'success' => true,
            'data' => $balance
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch balance'], 500);
    }
}

// Get deposit history for a user
function getDepositHistory() {
    global $pdo;
    
    $userId = $_GET['user_id'] ?? '';
    $limit = $_GET['limit'] ?? 20;
    $offset = $_GET['offset'] ?? 0;
    
    if (empty($userId)) {
        sendResponse(['error' => 'User ID is required'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT *
            FROM deposits
            WHERE user_id = ?
            ORDER BY created_at DESC
            LIMIT ? OFFSET ?
        ");
        $stmt->execute([$userId, $limit, $offset]);
        $deposits = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $deposits
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch deposit history'], 500);
    }
}

// Get pending deposits (for admin)
function getPendingDeposits() {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                d.*,
                u.username,
                u.email
            FROM deposits d
            LEFT JOIN users u ON d.user_id = u.id
            WHERE d.status = 'pending'
            ORDER BY d.created_at ASC
        ");
        $stmt->execute();
        $deposits = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $deposits
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch pending deposits'], 500);
    }
}

// Get available payment methods
function getPaymentMethods() {
    $methods = [
        [
            'id' => 'btc',
            'name' => 'Bitcoin',
            'symbol' => 'BTC',
            'min_amount' => 0.001,
            'max_amount' => 10.0,
            'fee_percentage' => 0,
            'processing_time' => '10-60 minutes',
            'status' => 'active'
        ],
        [
            'id' => 'eth',
            'name' => 'Ethereum',
            'symbol' => 'ETH',
            'min_amount' => 0.01,
            'max_amount' => 100.0,
            'fee_percentage' => 1.5,
            'processing_time' => '5-30 minutes',
            'status' => 'maintenance'
        ],
        [
            'id' => 'usdt',
            'name' => 'Tether',
            'symbol' => 'USDT',
            'min_amount' => 10.0,
            'max_amount' => 10000.0,
            'fee_percentage' => 1.0,
            'processing_time' => '5-15 minutes',
            'status' => 'active'
        ]
    ];
    
    sendResponse([
        'success' => true,
        'data' => $methods
    ]);
}

// Get current BTC rates
function getBtcRates() {
    // Mock BTC rates - in production, fetch from external API
    $rates = [
        'btc_usd' => 45000.00,
        'btc_eur' => 38000.00,
        'last_updated' => date('Y-m-d H:i:s'),
        'source' => 'coinbase'
    ];
    
    sendResponse([
        'success' => true,
        'data' => $rates
    ]);
}

// Create a new deposit
function createDeposit() {
    global $pdo, $input;
    
    validateRequired($input, ['user_id', 'amount_usd', 'payment_method']);
    
    $userId = $input['user_id'];
    $amountUsd = floatval($input['amount_usd']);
    $paymentMethod = $input['payment_method'];
    $cryptoAmount = $input['crypto_amount'] ?? null;
    $txHash = $input['tx_hash'] ?? null;
    
    if ($amountUsd <= 0) {
        sendResponse(['error' => 'Amount must be greater than 0'], 400);
    }
    
    try {
        $pdo->beginTransaction();
        
        // Generate deposit ID and address
        $depositId = generateId('dep_');
        $btcAddress = generateBtcAddress();
        
        // Insert deposit record
        $stmt = $pdo->prepare("
            INSERT INTO deposits (
                id, user_id, amount_usd, crypto_amount, payment_method, 
                btc_address, tx_hash, status, created_at
            )
            VALUES (?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        $stmt->execute([
            $depositId, $userId, $amountUsd, $cryptoAmount, 
            $paymentMethod, $btcAddress, $txHash
        ]);
        
        // Update user's pending balance
        $stmt = $pdo->prepare("
            UPDATE users 
            SET pending_balance = pending_balance + ? 
            WHERE id = ?
        ");
        $stmt->execute([$amountUsd, $userId]);
        
        $pdo->commit();
        
        sendResponse([
            'success' => true,
            'deposit_id' => $depositId,
            'btc_address' => $btcAddress,
            'message' => 'Deposit created successfully'
        ]);
    } catch(PDOException $e) {
        $pdo->rollback();
        sendResponse(['error' => 'Failed to create deposit'], 500);
    }
}

// Confirm a deposit (admin action)
function confirmDeposit() {
    global $pdo, $input;
    
    validateRequired($input, ['deposit_id', 'admin_id']);
    
    $depositId = $input['deposit_id'];
    $adminId = $input['admin_id'];
    $status = $input['status'] ?? 'confirmed'; // 'confirmed' or 'rejected'
    $adminNotes = $input['admin_notes'] ?? '';
    
    if (!in_array($status, ['confirmed', 'rejected'])) {
        sendResponse(['error' => 'Invalid status'], 400);
    }
    
    try {
        $pdo->beginTransaction();
        
        // Get deposit details
        $stmt = $pdo->prepare("SELECT * FROM deposits WHERE id = ? AND status = 'pending'");
        $stmt->execute([$depositId]);
        $deposit = $stmt->fetch();
        
        if (!$deposit) {
            sendResponse(['error' => 'Deposit not found or already processed'], 404);
        }
        
        // Update deposit status
        $stmt = $pdo->prepare("
            UPDATE deposits 
            SET status = ?, confirmed_by = ?, confirmed_at = NOW(), admin_notes = ?
            WHERE id = ?
        ");
        $stmt->execute([$status, $adminId, $adminNotes, $depositId]);
        
        if ($status === 'confirmed') {
            // Move from pending to confirmed balance
            $stmt = $pdo->prepare("
                UPDATE users 
                SET 
                    balance = balance + ?,
                    pending_balance = pending_balance - ?,
                    total_deposited = total_deposited + ?
                WHERE id = ?
            ");
            $stmt->execute([
                $deposit['amount_usd'], 
                $deposit['amount_usd'], 
                $deposit['amount_usd'], 
                $deposit['user_id']
            ]);
        } else {
            // Remove from pending balance
            $stmt = $pdo->prepare("
                UPDATE users 
                SET pending_balance = pending_balance - ?
                WHERE id = ?
            ");
            $stmt->execute([$deposit['amount_usd'], $deposit['user_id']]);
        }
        
        $pdo->commit();
        
        sendResponse([
            'success' => true,
            'message' => "Deposit {$status} successfully"
        ]);
    } catch(PDOException $e) {
        $pdo->rollback();
        sendResponse(['error' => 'Failed to confirm deposit'], 500);
    }
}

// Process a payment for a rental
function processPayment() {
    global $pdo, $input;
    
    validateRequired($input, ['user_id', 'amount', 'description']);
    
    $userId = $input['user_id'];
    $amount = floatval($input['amount']);
    $description = $input['description'];
    $rentalId = $input['rental_id'] ?? null;
    
    if ($amount <= 0) {
        sendResponse(['error' => 'Amount must be greater than 0'], 400);
    }
    
    try {
        $pdo->beginTransaction();
        
        // Check user balance
        $stmt = $pdo->prepare("SELECT balance FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if (!$user || $user['balance'] < $amount) {
            sendResponse(['error' => 'Insufficient balance'], 400);
        }
        
        // Deduct from balance
        $stmt = $pdo->prepare("
            UPDATE users 
            SET balance = balance - ?, total_withdrawn = total_withdrawn + ?
            WHERE id = ?
        ");
        $stmt->execute([$amount, $amount, $userId]);
        
        // Record transaction
        $transactionId = generateId('txn_');
        $stmt = $pdo->prepare("
            INSERT INTO transactions (
                id, user_id, amount, type, description, rental_id, status, created_at
            )
            VALUES (?, ?, ?, 'payment', ?, ?, 'completed', NOW())
        ");
        $stmt->execute([$transactionId, $userId, -$amount, $description, $rentalId]);
        
        $pdo->commit();
        
        sendResponse([
            'success' => true,
            'transaction_id' => $transactionId,
            'remaining_balance' => $user['balance'] - $amount,
            'message' => 'Payment processed successfully'
        ]);
    } catch(PDOException $e) {
        $pdo->rollback();
        sendResponse(['error' => 'Failed to process payment'], 500);
    }
}

// Generate BTC address (mock implementation)
function generateBtcAddress() {
    // In production, use proper BTC address generation
    $addresses = [
        '1A1zP1eP5QGefi2DMPTfTL5SLmv7DivfNa',
        '3J98t1WpEZ73CNmQviecrnyiWrnqRhWNLy',
        'bc1qxy2kgdygjrsqtzq2n0yrf2493p83kkfjhx0wlh',
        '1BvBMSEYstWetqTFn5Au4m4GFg7xJaNVN2',
        '3QJmV3qfvL9SuYo34YihAf3sRCW3qSinyC'
    ];
    
    return $addresses[array_rand($addresses)];
}

// Generate BTC address via API call
function generateBtcAddressAPI() {
    // Mock implementation - replace with actual address generation
    return generateBtcAddress();
}

// Update deposit status
function updateDepositStatus() {
    global $pdo, $input;
    
    validateRequired($input, ['deposit_id', 'status']);
    
    $depositId = $input['deposit_id'];
    $status = $input['status'];
    $txHash = $input['tx_hash'] ?? null;
    $blockConfirmations = $input['confirmations'] ?? 0;
    
    $validStatuses = ['pending', 'confirmed', 'rejected', 'expired'];
    if (!in_array($status, $validStatuses)) {
        sendResponse(['error' => 'Invalid status'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            UPDATE deposits 
            SET status = ?, tx_hash = ?, block_confirmations = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$status, $txHash, $blockConfirmations, $depositId]);
        
        sendResponse([
            'success' => true,
            'message' => 'Deposit status updated successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to update deposit status'], 500);
    }
}

// Update user balance directly (admin action)
function updateUserBalance() {
    global $pdo, $input;
    
    validateRequired($input, ['user_id', 'amount', 'type', 'description']);
    
    $userId = $input['user_id'];
    $amount = floatval($input['amount']);
    $type = $input['type']; // 'credit' or 'debit'
    $description = $input['description'];
    $adminId = $input['admin_id'] ?? null;
    
    if (!in_array($type, ['credit', 'debit'])) {
        sendResponse(['error' => 'Invalid type'], 400);
    }
    
    if ($amount <= 0) {
        sendResponse(['error' => 'Amount must be greater than 0'], 400);
    }
    
    try {
        $pdo->beginTransaction();
        
        // Update user balance
        $balanceChange = $type === 'credit' ? $amount : -$amount;
        $stmt = $pdo->prepare("
            UPDATE users 
            SET balance = balance + ?
            WHERE id = ?
        ");
        $stmt->execute([$balanceChange, $userId]);
        
        // Record transaction
        $transactionId = generateId('txn_');
        $stmt = $pdo->prepare("
            INSERT INTO transactions (
                id, user_id, amount, type, description, admin_id, status, created_at
            )
            VALUES (?, ?, ?, ?, ?, ?, 'completed', NOW())
        ");
        $stmt->execute([
            $transactionId, $userId, $balanceChange, 
            'adjustment', $description, $adminId
        ]);
        
        $pdo->commit();
        
        sendResponse([
            'success' => true,
            'transaction_id' => $transactionId,
            'message' => 'Balance updated successfully'
        ]);
    } catch(PDOException $e) {
        $pdo->rollback();
        sendResponse(['error' => 'Failed to update balance'], 500);
    }
}
?>