<?php
require_once 'config.php';

$method = $_SERVER['REQUEST_METHOD'];
$input = json_decode(file_get_contents('php://input'), true);

switch ($method) {
    case 'GET':
        handleGetRequests();
        break;
    case 'POST':
        handlePostRequests();
        break;
    case 'PUT':
        handlePutRequests();
        break;
    case 'DELETE':
        handleDeleteRequests();
        break;
    default:
        sendResponse(['error' => 'Method not allowed'], 405);
}

function handleGetRequests() {
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'asic_models':
            getAsicModels();
            break;
        case 'user_rentals':
            getUserRentals();
            break;
        case 'rental_details':
            getRentalDetails();
            break;
        case 'available_asics':
            getAvailableAsics();
            break;
        case 'rental_earnings':
            getRentalEarnings();
            break;
        case 'all_rentals':
            getAllRentals();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

function handlePostRequests() {
    global $input;
    $action = $input['action'] ?? '';
    
    switch ($action) {
        case 'create_rental':
            createRental();
            break;
        case 'calculate_earnings':
            calculateEarnings();
            break;
        case 'extend_rental':
            extendRental();
            break;
        case 'add_asic_model':
            addAsicModel();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

function handlePutRequests() {
    global $input;
    $action = $input['action'] ?? '';
    
    switch ($action) {
        case 'update_rental':
            updateRental();
            break;
        case 'update_asic_model':
            updateAsicModel();
            break;
        case 'confirm_rental':
            confirmRental();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

function handleDeleteRequests() {
    $action = $_GET['action'] ?? '';
    
    switch ($action) {
        case 'cancel_rental':
            cancelRental();
            break;
        default:
            sendResponse(['error' => 'Invalid action'], 400);
    }
}

// Get all ASIC models
function getAsicModels() {
    global $pdo;
    
    $category = $_GET['category'] ?? '';
    $available_only = $_GET['available_only'] ?? false;
    
    try {
        $sql = "SELECT * FROM asic_models WHERE 1=1";
        $params = [];
        
        if (!empty($category)) {
            $sql .= " AND category = ?";
            $params[] = $category;
        }
        
        if ($available_only) {
            $sql .= " AND status = 'active' AND available_units > 0";
        }
        
        $sql .= " ORDER BY name ASC";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $models = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $models
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch ASIC models'], 500);
    }
}

// Get user's rentals
function getUserRentals() {
    global $pdo;
    
    $userId = $_GET['user_id'] ?? '';
    
    if (empty($userId)) {
        sendResponse(['error' => 'User ID is required'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                r.*,
                am.name as asic_name,
                am.hashrate,
                am.power_consumption,
                am.image_url
            FROM rentals r
            LEFT JOIN asic_models am ON r.asic_model_id = am.id
            WHERE r.user_id = ?
            ORDER BY r.created_at DESC
        ");
        $stmt->execute([$userId]);
        $rentals = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $rentals
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch user rentals'], 500);
    }
}

// Get rental details
function getRentalDetails() {
    global $pdo;
    
    $rentalId = $_GET['rental_id'] ?? '';
    
    if (empty($rentalId)) {
        sendResponse(['error' => 'Rental ID is required'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                r.*,
                am.name as asic_name,
                am.hashrate,
                am.power_consumption,
                am.daily_earnings_estimate,
                am.image_url,
                u.username,
                u.email
            FROM rentals r
            LEFT JOIN asic_models am ON r.asic_model_id = am.id
            LEFT JOIN users u ON r.user_id = u.id
            WHERE r.id = ?
        ");
        $stmt->execute([$rentalId]);
        $rental = $stmt->fetch();
        
        if (!$rental) {
            sendResponse(['error' => 'Rental not found'], 404);
        }
        
        sendResponse([
            'success' => true,
            'data' => $rental
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch rental details'], 500);
    }
}

// Get available ASICs for rent
function getAvailableAsics() {
    global $pdo;
    
    $priceMin = $_GET['price_min'] ?? 0;
    $priceMax = $_GET['price_max'] ?? 999999;
    $hashrateMin = $_GET['hashrate_min'] ?? 0;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                *,
                (daily_price * 30) as monthly_price,
                (daily_earnings_estimate - daily_price) as daily_profit
            FROM asic_models 
            WHERE status = 'active' 
                AND available_units > 0
                AND daily_price BETWEEN ? AND ?
                AND hashrate >= ?
            ORDER BY daily_profit DESC, hashrate DESC
        ");
        $stmt->execute([$priceMin, $priceMax, $hashrateMin]);
        $asics = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $asics
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch available ASICs'], 500);
    }
}

// Get rental earnings
function getRentalEarnings() {
    global $pdo;
    
    $rentalId = $_GET['rental_id'] ?? '';
    
    if (empty($rentalId)) {
        sendResponse(['error' => 'Rental ID is required'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                DATE(created_at) as date,
                SUM(amount) as total_earned,
                COUNT(*) as payment_count
            FROM mining_rewards
            WHERE rental_id = ?
            GROUP BY DATE(created_at)
            ORDER BY date DESC
            LIMIT 30
        ");
        $stmt->execute([$rentalId]);
        $earnings = $stmt->fetchAll();
        
        // Get total earnings
        $stmt = $pdo->prepare("
            SELECT 
                SUM(amount) as total_earnings,
                COUNT(*) as total_payments,
                MIN(created_at) as first_payment,
                MAX(created_at) as last_payment
            FROM mining_rewards
            WHERE rental_id = ?
        ");
        $stmt->execute([$rentalId]);
        $summary = $stmt->fetch();
        
        sendResponse([
            'success' => true,
            'data' => [
                'daily_earnings' => $earnings,
                'summary' => $summary
            ]
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch rental earnings'], 500);
    }
}

// Get all rentals (admin)
function getAllRentals() {
    global $pdo;
    
    $status = $_GET['status'] ?? '';
    $limit = $_GET['limit'] ?? 50;
    $offset = $_GET['offset'] ?? 0;
    
    try {
        $sql = "
            SELECT 
                r.*,
                u.username,
                u.email,
                am.name as asic_name
            FROM rentals r
            LEFT JOIN users u ON r.user_id = u.id
            LEFT JOIN asic_models am ON r.asic_model_id = am.id
            WHERE 1=1
        ";
        $params = [];
        
        if (!empty($status)) {
            $sql .= " AND r.status = ?";
            $params[] = $status;
        }
        
        $sql .= " ORDER BY r.created_at DESC LIMIT ? OFFSET ?";
        $params[] = $limit;
        $params[] = $offset;
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        $rentals = $stmt->fetchAll();
        
        sendResponse([
            'success' => true,
            'data' => $rentals
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to fetch all rentals'], 500);
    }
}

// Create a new rental
function createRental() {
    global $pdo, $input;
    
    validateRequired($input, ['user_id', 'asic_model_id', 'duration_days']);
    
    $userId = $input['user_id'];
    $asicModelId = $input['asic_model_id'];
    $durationDays = intval($input['duration_days']);
    $quantity = intval($input['quantity'] ?? 1);
    
    if ($durationDays <= 0 || $quantity <= 0) {
        sendResponse(['error' => 'Duration and quantity must be greater than 0'], 400);
    }
    
    try {
        $pdo->beginTransaction();
        
        // Get ASIC model details
        $stmt = $pdo->prepare("SELECT * FROM asic_models WHERE id = ? AND status = 'active'");
        $stmt->execute([$asicModelId]);
        $asicModel = $stmt->fetch();
        
        if (!$asicModel) {
            sendResponse(['error' => 'ASIC model not found or not available'], 404);
        }
        
        if ($asicModel['available_units'] < $quantity) {
            sendResponse(['error' => 'Insufficient available units'], 400);
        }
        
        // Calculate total cost
        $totalCost = $asicModel['daily_price'] * $durationDays * $quantity;
        
        // Check user balance
        $stmt = $pdo->prepare("SELECT balance FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        if (!$user || $user['balance'] < $totalCost) {
            sendResponse(['error' => 'Insufficient balance'], 400);
        }
        
        // Create rental
        $rentalId = generateId('rent_');
        $startDate = date('Y-m-d H:i:s');
        $endDate = date('Y-m-d H:i:s', strtotime("+{$durationDays} days"));
        
        $stmt = $pdo->prepare("
            INSERT INTO rentals (
                id, user_id, asic_model_id, quantity, duration_days, 
                start_date, end_date, daily_rate, total_cost, status, created_at
            )
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())
        ");
        $stmt->execute([
            $rentalId, $userId, $asicModelId, $quantity, $durationDays,
            $startDate, $endDate, $asicModel['daily_price'], $totalCost
        ]);
        
        // Deduct from user balance
        $stmt = $pdo->prepare("
            UPDATE users 
            SET balance = balance - ?
            WHERE id = ?
        ");
        $stmt->execute([$totalCost, $userId]);
        
        // Update available units
        $stmt = $pdo->prepare("
            UPDATE asic_models 
            SET available_units = available_units - ?
            WHERE id = ?
        ");
        $stmt->execute([$quantity, $asicModelId]);
        
        // Record transaction
        $transactionId = generateId('txn_');
        $stmt = $pdo->prepare("
            INSERT INTO transactions (
                id, user_id, amount, type, description, rental_id, status, created_at
            )
            VALUES (?, ?, ?, 'rental_payment', ?, ?, 'completed', NOW())
        ");
        $stmt->execute([
            $transactionId, $userId, -$totalCost, 
            "Rental payment for {$asicModel['name']}", $rentalId
        ]);
        
        $pdo->commit();
        
        sendResponse([
            'success' => true,
            'rental_id' => $rentalId,
            'total_cost' => $totalCost,
            'message' => 'Rental created successfully'
        ]);
    } catch(PDOException $e) {
        $pdo->rollback();
        sendResponse(['error' => 'Failed to create rental'], 500);
    }
}

// Calculate earnings estimate
function calculateEarnings() {
    global $input;
    
    validateRequired($input, ['asic_model_id', 'duration_days', 'quantity']);
    
    $asicModelId = $input['asic_model_id'];
    $durationDays = intval($input['duration_days']);
    $quantity = intval($input['quantity']);
    
    // Mock calculation - in production, use real mining data
    $dailyEarnings = 15.50; // USD per day per unit
    $dailyCost = 12.00; // USD per day per unit
    $dailyProfit = $dailyEarnings - $dailyCost;
    
    $totalEarnings = $dailyEarnings * $durationDays * $quantity;
    $totalCost = $dailyCost * $durationDays * $quantity;
    $totalProfit = $totalEarnings - $totalCost;
    $roi = ($totalProfit / $totalCost) * 100;
    
    sendResponse([
        'success' => true,
        'data' => [
            'daily_earnings' => $dailyEarnings,
            'daily_cost' => $dailyCost,
            'daily_profit' => $dailyProfit,
            'total_earnings' => $totalEarnings,
            'total_cost' => $totalCost,
            'total_profit' => $totalProfit,
            'roi_percentage' => round($roi, 2),
            'break_even_days' => round($totalCost / $dailyProfit, 1)
        ]
    ]);
}

// Extend rental duration
function extendRental() {
    global $pdo, $input;
    
    validateRequired($input, ['rental_id', 'additional_days']);
    
    $rentalId = $input['rental_id'];
    $additionalDays = intval($input['additional_days']);
    $userId = $input['user_id'] ?? '';
    
    if ($additionalDays <= 0) {
        sendResponse(['error' => 'Additional days must be greater than 0'], 400);
    }
    
    try {
        $pdo->beginTransaction();
        
        // Get rental details
        $stmt = $pdo->prepare("
            SELECT r.*, am.daily_price 
            FROM rentals r
            LEFT JOIN asic_models am ON r.asic_model_id = am.id
            WHERE r.id = ? AND r.status = 'active'
        ");
        $stmt->execute([$rentalId]);
        $rental = $stmt->fetch();
        
        if (!$rental) {
            sendResponse(['error' => 'Rental not found or not active'], 404);
        }
        
        if (!empty($userId) && $rental['user_id'] !== $userId) {
            sendResponse(['error' => 'Unauthorized access'], 403);
        }
        
        // Calculate extension cost
        $extensionCost = $rental['daily_price'] * $additionalDays * $rental['quantity'];
        
        // Check user balance
        $stmt = $pdo->prepare("SELECT balance FROM users WHERE id = ?");
        $stmt->execute([$rental['user_id']]);
        $user = $stmt->fetch();
        
        if ($user['balance'] < $extensionCost) {
            sendResponse(['error' => 'Insufficient balance for extension'], 400);
        }
        
        // Update rental
        $newEndDate = date('Y-m-d H:i:s', strtotime($rental['end_date'] . " +{$additionalDays} days"));
        $newDuration = $rental['duration_days'] + $additionalDays;
        $newTotalCost = $rental['total_cost'] + $extensionCost;
        
        $stmt = $pdo->prepare("
            UPDATE rentals 
            SET 
                duration_days = ?,
                end_date = ?,
                total_cost = ?,
                updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$newDuration, $newEndDate, $newTotalCost, $rentalId]);
        
        // Deduct extension cost
        $stmt = $pdo->prepare("
            UPDATE users 
            SET balance = balance - ?
            WHERE id = ?
        ");
        $stmt->execute([$extensionCost, $rental['user_id']]);
        
        // Record transaction
        $transactionId = generateId('txn_');
        $stmt = $pdo->prepare("
            INSERT INTO transactions (
                id, user_id, amount, type, description, rental_id, status, created_at
            )
            VALUES (?, ?, ?, 'rental_extension', ?, ?, 'completed', NOW())
        ");
        $stmt->execute([
            $transactionId, $rental['user_id'], -$extensionCost,
            "Rental extension for {$additionalDays} days", $rentalId
        ]);
        
        $pdo->commit();
        
        sendResponse([
            'success' => true,
            'extension_cost' => $extensionCost,
            'new_end_date' => $newEndDate,
            'message' => 'Rental extended successfully'
        ]);
    } catch(PDOException $e) {
        $pdo->rollback();
        sendResponse(['error' => 'Failed to extend rental'], 500);
    }
}

// Add new ASIC model (admin)
function addAsicModel() {
    global $pdo, $input;
    
    validateRequired($input, ['name', 'hashrate', 'power_consumption', 'daily_price', 'available_units']);
    
    $name = $input['name'];
    $hashrate = $input['hashrate'];
    $powerConsumption = $input['power_consumption'];
    $dailyPrice = floatval($input['daily_price']);
    $availableUnits = intval($input['available_units']);
    $category = $input['category'] ?? 'bitcoin';
    $algorithm = $input['algorithm'] ?? 'SHA-256';
    $manufacturer = $input['manufacturer'] ?? '';
    $dailyEarnings = floatval($input['daily_earnings_estimate'] ?? $dailyPrice * 1.3);
    $imageUrl = $input['image_url'] ?? '/images/asic-miner.png';
    
    try {
        $asicId = generateId('asic_');
        $stmt = $pdo->prepare("
            INSERT INTO asic_models (
                id, name, hashrate, power_consumption, daily_price, 
                available_units, category, algorithm, manufacturer,
                daily_earnings_estimate, image_url, status, created_at
            )
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'active', NOW())
        ");
        $stmt->execute([
            $asicId, $name, $hashrate, $powerConsumption, $dailyPrice,
            $availableUnits, $category, $algorithm, $manufacturer,
            $dailyEarnings, $imageUrl
        ]);
        
        sendResponse([
            'success' => true,
            'asic_id' => $asicId,
            'message' => 'ASIC model added successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to add ASIC model'], 500);
    }
}

// Update rental status
function updateRental() {
    global $pdo, $input;
    
    validateRequired($input, ['rental_id', 'status']);
    
    $rentalId = $input['rental_id'];
    $status = $input['status'];
    $adminId = $input['admin_id'] ?? null;
    
    $validStatuses = ['pending', 'active', 'completed', 'cancelled'];
    if (!in_array($status, $validStatuses)) {
        sendResponse(['error' => 'Invalid status'], 400);
    }
    
    try {
        $stmt = $pdo->prepare("
            UPDATE rentals 
            SET status = ?, updated_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$status, $rentalId]);
        
        sendResponse([
            'success' => true,
            'message' => 'Rental status updated successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to update rental status'], 500);
    }
}

// Confirm rental (admin)
function confirmRental() {
    global $pdo, $input;
    
    validateRequired($input, ['rental_id', 'admin_id']);
    
    $rentalId = $input['rental_id'];
    $adminId = $input['admin_id'];
    
    try {
        $stmt = $pdo->prepare("
            UPDATE rentals 
            SET status = 'active', confirmed_by = ?, confirmed_at = NOW()
            WHERE id = ? AND status = 'pending'
        ");
        $stmt->execute([$adminId, $rentalId]);
        
        if ($stmt->rowCount() === 0) {
            sendResponse(['error' => 'Rental not found or already processed'], 404);
        }
        
        sendResponse([
            'success' => true,
            'message' => 'Rental confirmed successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to confirm rental'], 500);
    }
}

// Cancel rental
function cancelRental() {
    $rentalId = $_GET['rental_id'] ?? '';
    $userId = $_GET['user_id'] ?? '';
    
    if (empty($rentalId)) {
        sendResponse(['error' => 'Rental ID is required'], 400);
    }
    
    // Implementation for rental cancellation
    sendResponse([
        'success' => true,
        'message' => 'Rental cancellation processed'
    ]);
}

// Update ASIC model
function updateAsicModel() {
    global $pdo, $input;
    
    validateRequired($input, ['asic_id']);
    
    $asicId = $input['asic_id'];
    $updates = [];
    $params = [];
    
    $allowedFields = [
        'name', 'hashrate', 'power_consumption', 'daily_price', 
        'available_units', 'daily_earnings_estimate', 'status'
    ];
    
    foreach ($allowedFields as $field) {
        if (isset($input[$field])) {
            $updates[] = "$field = ?";
            $params[] = $input[$field];
        }
    }
    
    if (empty($updates)) {
        sendResponse(['error' => 'No valid fields to update'], 400);
    }
    
    $params[] = $asicId;
    
    try {
        $sql = "UPDATE asic_models SET " . implode(', ', $updates) . ", updated_at = NOW() WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
        
        sendResponse([
            'success' => true,
            'message' => 'ASIC model updated successfully'
        ]);
    } catch(PDOException $e) {
        sendResponse(['error' => 'Failed to update ASIC model'], 500);
    }
}
?>